<?php
/**
 * This file is part of backups plugin for FacturaScripts
 * Copyright (C) 2020-2021 Carlos Garcia Gomez <neorazorx@gmail.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
require __DIR__ . '/../vendor/autoload.php';

use Coderatio\SimpleBackup\SimpleBackup;

/**
 * Description of backups_wizard
 *
 * @author Carlos Garcia Gomez <neorazorx@gmail.com>
 */
class backups_wizard extends fs_controller
{

    public function __construct()
    {
        parent::__construct(__CLASS__, 'Backups', 'admin', true, true);
    }

    private function download_db_action()
    {
        /// disable template
        $this->template = false;

        SimpleBackup::setDatabase([FS_DB_NAME, FS_DB_USER, FS_DB_PASS, FS_DB_HOST])->downloadAfterExport(FS_DB_NAME);
    }

    private function download_files_action()
    {
        $folder = empty(FS_MYDOCS) ? FS_FOLDER : FS_MYDOCS;
        $file_name = FS_DB_NAME . '.zip';
        if ($this->zip($folder, $file_name)) {
            $this->template = false;
            header('Content-Type: ' . mime_content_type($file_name));
            header('Content-Disposition: attachment; filename="' . $file_name . '"');
            readfile($file_name);
            unlink($file_name);
            return;
        }
    }

    protected function private_core()
    {
        parent::private_core();

        if (strtolower(FS_DB_TYPE) !== 'mysql') {
            $this->new_error_msg('Este plugin solamente incluye soporte para MySQL');
            return;
        }

        $action = isset($_REQUEST['action']) ? $_REQUEST['action'] : '';
        switch ($action) {
            case 'download-db':
                $this->download_db_action();
                break;

            case 'download-files':
                $this->download_files_action();
                break;

            case 'restore':
                $this->restore_action();
                break;
        }
    }

    private function restore_action()
    {
        $this->restore_db_action();
        $this->restore_files_action();
    }

    private function restore_db_action()
    {
        if (false === is_uploaded_file($_FILES['dbfile']['tmp_name'])) {
            $this->new_error_msg('Error al restaurar la base de datos');
            return;
        }

        $this->db->close();
        $backup = SimpleBackup::setDatabase([FS_DB_NAME, FS_DB_USER, FS_DB_PASS, FS_DB_HOST])->importFrom($_FILES['dbfile']['tmp_name']);
        if (false === $backup->getResponse()->status) {
            $this->new_error_msg('Error al restaurar la base de datos');
            $this->db->connect();
            return;
        }

        $this->new_message('Base de datos restaurada correctamente');
        $this->db->connect();
    }

    private function restore_files_action()
    {
        foreach (fs_file_manager::not_writable_folders() as $folder) {
            $this->new_error_msg('FacturaScripts no tiene permisos para escribir en la carpeta: ' . $folder);
            return;
        }

        if (false === is_uploaded_file($_FILES['fsfile']['tmp_name'])) {
            $this->new_error_msg('Error al restaurar los archivos');
            return;
        }

        $zip = new ZipArchive();
        $zip_status = $zip->open($_FILES['fsfile']['tmp_name'], ZipArchive::CHECKCONS);
        if ($zip_status !== TRUE) {
            $this->core_log->new_error('Error al comprobar el archivo zip. Código: ' . $zip_status);
            return;
        }

        $folder = empty(FS_MYDOCS) ? FS_FOLDER : FS_MYDOCS;
        rename($folder . '/config.php', $folder . '/config-old.php');
        $zip->extractTo($folder);
        unlink($folder . '/config.php');
        rename($folder . '/config-old.php', $folder . '/config.php');
        $zip->close();
    }

    /**
     * 
     * @param string $source
     * @param string $destination
     *
     * @return bool
     */
    private function zip($source, $destination)
    {
        if (false === extension_loaded('zip') || false === file_exists($source)) {
            return false;
        }

        $zip = new ZipArchive();
        if (false === $zip->open($destination, ZIPARCHIVE::CREATE | ZipArchive::OVERWRITE)) {
            return false;
        }

        $files = new RecursiveIteratorIterator(
            new RecursiveDirectoryIterator($source),
            RecursiveIteratorIterator::LEAVES_ONLY
        );

        foreach ($files as $name => $file) {
            if ($file->isDir() || substr($name, -4) === '.zip') {
                continue;
            }

            $filePath = $file->getRealPath();
            $relativePath = substr($filePath, strlen($source) + 1);

            $zip->addFile($filePath, $relativePath);
        }

        return $zip->close();
    }
}
